/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include <udm_config.h>

#include <stdio.h>
#include <stdlib.h>

#ifdef HAVE_ZLIB
#include <zlib.h>

#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <errno.h>
#include <unistd.h>
#include <signal.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <netdb.h>
#include <math.h>
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#include "udm_common.h"
#include "udm_utils.h"
#include "udm_env.h"
#include "udm_agent.h"
#include "udm_conf.h"
#include "udm_log.h"
#include "udm_services.h"
#include "udm_store.h"
#include "udm_vars.h"
#include "udm_xmalloc.h"
#include "udm_vars.h"
#include "udm_parsehtml.h"
#include "udm_searchtool.h"

#define UDM_STORED_MAXCLNTS 128

static char pidname[1024];
static char time_pid[100];
static UDM_ENV * Conf = NULL;
static UDM_AGENT * Agent = NULL;
static int verb = -1;


static void exitproc(void){
	unlink(pidname);
}

static char * time_pid_info(void){
	struct tm * tim;
	time_t t;
	t=time(NULL);
	tim=localtime(&t);
	strftime(time_pid,sizeof(time_pid),"%a %d %T",tim);
	sprintf(time_pid+strlen(time_pid)," [%d]",(int)getpid());
	return(time_pid);
}

/********************* SIG Handlers ****************/

static void sighandler(int sign);
static void Optimize_sighandler(int sign);

static void init_signals(void){
	/* Set up signals handler*/
	UdmSignal(SIGPIPE, sighandler);
	UdmSignal(SIGHUP, sighandler);
	UdmSignal(SIGINT, sighandler);
	UdmSignal(SIGTERM, sighandler);
	UdmSignal(SIGCHLD, sighandler);
	UdmSignal(SIGALRM, sighandler);
}
static void Optimize_init_signals(void){
	/* Set up signals handler*/
	UdmSignal(SIGPIPE, Optimize_sighandler);
	UdmSignal(SIGHUP, Optimize_sighandler);
	UdmSignal(SIGINT, Optimize_sighandler);
	UdmSignal(SIGTERM, Optimize_sighandler);
	UdmSignal(SIGCHLD, Optimize_sighandler);
	UdmSignal(SIGALRM, Optimize_sighandler);
}
static int have_sighup=0;
static int have_sigint=0;
static int have_sigterm=0;
static int have_sigpipe=0;
static int have_sigalrm=0;

static void sighandler(int sign){
#ifdef UNIONWAIT
	union wait status;
#else
	int status;
#endif

	switch(sign){
		case SIGPIPE:
		  have_sigpipe = 1;
		  break;
		case SIGHUP:
		  have_sighup = 1;
		  break;
	        case SIGCHLD:
		  while (waitpid(-1, &status, WNOHANG) > 0);
		  break;
		case SIGINT:
		  have_sigint=1;
		  break;
		case SIGTERM:
		  have_sigterm=1;
		  break;
	        case SIGALRM:
		  _exit(0);
		  break;
		default: ;
	}
}

static void Optimize_sighandler(int sign){
#ifdef UNIONWAIT
	union wait status;
#else
	int status;
#endif

	switch(sign){
		case SIGPIPE:
		  have_sigpipe = 1;
		  break;
		case SIGHUP:
		  have_sighup = 1;
		  break;
	        case SIGCHLD:
		  wait(&status);
		  break;
		case SIGINT:
		  have_sigint = 1;
		  break;
		case SIGTERM:
		  have_sigterm = 1;
		  break;
	        case SIGALRM:
		  have_sigalrm = 1;
		  break;
		default: ;
	}
}


#define UDM_STOREIND_SIG "msINDEX\0"
#define UDM_STORE_SIG "msSTORE\0"
#define UDM_SIG_LEN 8
#define UDM_STORE_BITS 0xFFFF      /* bits of rec_id for file no. */

#define UDM_HASH_PRIME 4093
#define UDM_HASH(x)    ((x) % UDM_HASH_PRIME)

typedef struct StoreItem {
  unsigned int rec_id;
  long offset;
  long next;
  size_t size;
} UDM_STOREITEM;

typedef struct SortStoreItem {
  long offset;
  UDM_STOREITEM Item;
} UDM_SORTSTOREITEM;


static int cmpsi(const void *s1, const void *s2) {
  const UDM_SORTSTOREITEM *d1 = (const UDM_SORTSTOREITEM*)s1;
  const UDM_SORTSTOREITEM *d2 = (const UDM_SORTSTOREITEM*)s2;
  if ((d1->Item.offset) < (d2->Item.offset)) return -1;
  if ((d1->Item.offset) > (d2->Item.offset)) return 1;
  return 0;
}

static struct flock* file_lock(struct flock *ret, int type, int whence) {
    ret->l_type = type ;
    ret->l_start = 0 ;
    ret->l_whence = whence ;
    ret->l_len = 0 ;
    ret->l_pid = getpid() ;
    return ret ;
}

static void read_lock(FILE *f) {   /* a shared lock on an entire file */
  struct flock ret;

  fcntl(fileno(f), F_SETLKW, file_lock(&ret, F_RDLCK, SEEK_SET));
}

static void write_lock(FILE *f) {  /* an exclusive lock on an entire file */
  struct flock ret;

  fcntl(fileno(f), F_SETLKW, file_lock(&ret, F_WRLCK, SEEK_SET));
}

static void un_lock(FILE *f) { /* ulock an entire file */
  struct flock ret;
  fcntl(fileno(f), F_SETLKW, file_lock(&ret, F_UNLCK, SEEK_SET));
}


const char *store_dir;
unsigned int FileNo, hash, mishash = 0;
size_t  DocSize = 0, filenamelen;
char *Ifilename = NULL, *Sfilename = NULL;
Byte *Doc = NULL, *CDoc = NULL;
FILE  *Ifd = NULL, *Sfd = NULL;
int ns, StoredFiles, OptimizeInterval, OptimizeRatio;
char sig[8];
UDM_STOREITEM  Item, *hTable;
long CurrentItemPos;


static void CloseBase(void) {
  fflush(Sfd);
  fflush(Ifd);
  un_lock(Sfd); 
  un_lock(Ifd); 
  fclose(Sfd); 
  fclose(Ifd); 
  UDM_FREE(Ifilename);
  UDM_FREE(Sfilename);
  return;
}

#define ABORT	    UDM_FREE(Doc); \
                    UDM_FREE(CDoc); \
                    CloseBase(); \
		    exit (0);
#define UDM_READ_LOCK  0
#define UDM_WRITE_LOCK 1


static void OpenBase(unsigned int rec_id, int mode) {
		  FileNo =  (rec_id & UDM_STORE_BITS) % UdmVarListFindInt(&Conf->Vars, "StoredFiles", 0x10000);
		  hash = UDM_HASH(rec_id);
		  filenamelen = ((Conf->vardir == NULL) ? strlen(UDM_VAR_DIR "/store/") : strlen(Conf->vardir) + 7) + 34;
		  store_dir = (Conf->vardir == NULL) ? UDM_VAR_DIR : Conf->vardir;
		  if (
		      ((Ifilename = (char *)UdmXmalloc(filenamelen)) == NULL) ||
		      ((Sfilename = (char *)UdmXmalloc(filenamelen)) == NULL)		      ) {
		    UDM_FREE(Doc);
		    UDM_FREE(CDoc);
                    UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
                    closesocket(ns); 
		    exit(0);
		  }
		  sprintf(Sfilename, "%s/store/%04x.s", store_dir, FileNo);
		  sprintf(Ifilename, "%s/store/%04x.i", store_dir, FileNo);

		  if ((Ifd = fopen(Ifilename, "rb+")) == NULL) {
		    if ((Ifd = fopen(Ifilename, "wb+")) == NULL) {
		      fprintf(stderr, "Can't open/create file %s", Ifilename);
		      UDM_FREE(Doc); 
		      UDM_FREE(CDoc);
		      UDM_FREE(Ifilename); 
		      UDM_FREE(Sfilename); 
		      UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
		      closesocket(ns); 
		      exit (0);
		    }
		    if ((hTable = (UDM_STOREITEM *)UdmXmalloc(sizeof(UDM_STOREITEM) * UDM_HASH_PRIME)) == NULL) {
		      UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
		      closesocket(ns); 
		      ABORT;
		    }
		    bzero(hTable, sizeof(UDM_STOREITEM) * UDM_HASH_PRIME);
		    if (
			(fwrite(UDM_STOREIND_SIG, UDM_SIG_LEN, 1, Ifd) != 1) ||
			(fwrite(hTable, sizeof(UDM_STOREITEM), UDM_HASH_PRIME, Ifd) != UDM_HASH_PRIME) 
			) {
		      fprintf(stderr, "Can't set new index for file %s\n", Ifilename);
		      UDM_FREE(hTable);
		      UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
		      closesocket(ns); 
		      ABORT;
		    }
		    UDM_FREE(hTable);
		    if (fseek(Ifd, 0, SEEK_SET)) {
		      UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
		      closesocket(ns); 
		      ABORT;
		    }
		  }
		  switch (mode) {
		  case UDM_READ_LOCK:
		    read_lock(Ifd);
		    break;
		  case UDM_WRITE_LOCK:
		    write_lock(Ifd);
		    break;
		  }

		  if ((Sfd = fopen(Sfilename, "rb+")) == NULL) {
		    if ((Sfd = fopen(Sfilename, "wb+")) == NULL) {
		      fprintf(stderr, "Can't open/create file %s", Sfilename);
		      UDM_FREE(Doc); 
		      UDM_FREE(CDoc);
		      UDM_FREE(Ifilename); 
		      UDM_FREE(Sfilename);
		      un_lock(Ifd);
		      fclose(Ifd);
		      UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
		      closesocket(ns); 
		      exit (0);
		    }
		    if (fwrite(UDM_STORE_SIG, UDM_SIG_LEN, 1, Sfd) != 1) { 
		      fprintf(stderr, "Can't set signature for file %s\n", Sfilename);
		      UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
		      closesocket(ns); 
		      ABORT;
		    }
		    if (fseek(Sfd, 0, SEEK_SET)) {
		      UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
		      closesocket(ns); 
		      ABORT;
		    }
		  }
		  switch(mode) {
		  case UDM_READ_LOCK:
		    read_lock(Sfd);
		    break;
		  case UDM_WRITE_LOCK:
		    write_lock(Sfd);
		    break;
		  }

		  /* check store signature or set for new */
		  if (fread(sig, UDM_SIG_LEN, 1, Sfd) != 1) {
		    fprintf(stderr, "Can't read signature for file %s\n", Sfilename);
		    UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
		    closesocket(ns); 
		    ABORT;
		  }
		  if ( strcmp(sig, UDM_STORE_SIG) != 0 ) {
		    fprintf(stderr, "Can't check signature %s for file %s\n", sig, Sfilename);
		    UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
		    closesocket(ns); 
		    ABORT;
		  }
		  /* check index signature or set for new */
		  if  (fread(sig, UDM_SIG_LEN, 1, Ifd) != 1) {/* can't read signature, try to write one */
		    fprintf(stderr, "Can't read signature for file %s\n", Ifilename);
		    UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
		    closesocket(ns); 
		    ABORT;
		  }
		  if ( strcmp(sig, UDM_STOREIND_SIG) != 0 ) {
		    fprintf(stderr, "Can't check signature for file %s\n", Ifilename);
		    UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
		    closesocket(ns); 
		    ABORT;
		  }

		    /* search rec_id */

		    if (fseek(Ifd, (long)(UDM_SIG_LEN + hash * sizeof(UDM_STOREITEM)), SEEK_SET)) {
		      ABORT;
		    }
		    CurrentItemPos = ftell(Ifd);
		    if (fread(&Item, sizeof(UDM_STOREITEM), 1, Ifd) != 1) {
		      fprintf(stderr, "Can't read index for file %s %ld\n", Ifilename, CurrentItemPos);
		      ABORT;
		    }

		    mishash = 0;
		    while((Item.next != 0) && (Item.rec_id != rec_id)) {
		      CurrentItemPos = Item.next;
		      if (fseek(Ifd, CurrentItemPos, SEEK_SET)) {
			ABORT;
		      }
		      if (fread(&Item, sizeof(UDM_STOREITEM), 1, Ifd) != 1) {
			fprintf(stderr, "Can't read hash chain for file %s\n", Ifilename);
			ABORT;
		      }
		      mishash = 1;
		    }

}

static void Optimize(void) {
  unsigned int current_base = 0;

	Optimize_init_signals();
	alarm((unsigned int) OptimizeInterval); /* try optimize one base every OptimizeInterval sec. */
	while(1) {
	  sleep(1);
		if(have_sigint){
			UdmLog(Agent, verb, "Optimize: SIGINT arrived");
			have_sigint = 0;
			break;
		}
		if(have_sigterm){
			UdmLog(Agent, verb, "Optimize: SIGTERM arrived");
			have_sigterm = 0;
			break;
		}
		if(have_sigpipe){
			UdmLog(Agent,verb,"Optimize: SIGPIPE arrived. Broken pipe !");
			have_sigpipe = 0;
			break;
		}
		if (have_sigalrm) {
		        long ActualSize = 0, SSize, diff, gain, i, j, nitems = 0, pos, posold;
			size_t nread;
			double dr = 0;
			char buffer[BUFSIZ];
			OpenBase(current_base, UDM_WRITE_LOCK);
			fseek(Ifd, UDM_SIG_LEN, SEEK_SET);
			while(fread(&Item, sizeof(UDM_STOREITEM), 1, Ifd) == 1) {
			  if (Item.rec_id != 0) {
			    ActualSize += (long)Item.size;
			    nitems++;
			  }
			}
			fseek(Sfd, 0, SEEK_END);
			SSize = ftell(Sfd) - UDM_SIG_LEN;
			dr = (nitems) ? fabs(100.0 * (SSize - ActualSize) / SSize) : 0.0,
			UdmLog(Agent,verb,"Optimize: base 0x%X, %ld files stored, defragmentation: %5.2f%%, Data size: %ld File size: %ld", current_base,
			       nitems, dr,
			       ActualSize, SSize
			       );
			if (dr > (float)OptimizeRatio) {
			  UDM_SORTSTOREITEM *si = (UDM_SORTSTOREITEM*)malloc(nitems * sizeof(UDM_SORTSTOREITEM));
			  fseek(Ifd, UDM_SIG_LEN, SEEK_SET);
			  i = 0;
			  pos = UDM_SIG_LEN;
			  while(fread(&Item, sizeof(UDM_STOREITEM), 1, Ifd) == 1) {
			    if(Item.rec_id != 0) {
			      si[i].offset = pos;
			      memcpy(&si[i].Item, &Item, sizeof(UDM_STOREITEM));
			      i++;
			    }
			    pos = ftell(Ifd);
			  }
#ifdef HAVE_MERGESORT
			  mergesort((void*)si, (size_t)nitems, sizeof(UDM_SORTSTOREITEM), cmpsi);
#else
			  qsort((void*)si, (size_t)nitems, sizeof(UDM_SORTSTOREITEM), cmpsi);
#endif
			  gain = 0;
			  pos = UDM_SIG_LEN;
			  posold = (nitems) ? si[0].Item.offset : UDM_SIG_LEN;
			  if ((diff = (posold - pos)) > 0) {
			    for(
				fseek(Sfd, posold, SEEK_SET);
				(nread = fread(buffer, 1, BUFSIZ, Sfd)) > 0;
				fseek(Sfd, posold, SEEK_SET)
				) {
			      fseek(Sfd, pos, SEEK_SET);
			      fwrite(buffer, 1, nread, Sfd);
			      posold += (long)nread;
			      pos += (long)nread;
			    }
			    fseek(Sfd, 0, SEEK_END);
			    SSize = ftell(Sfd);
			    ftruncate(fileno(Sfd), (off_t)(SSize - diff));
			    for (j = 0; j < nitems; j++) si[j].Item.offset -= diff;
			    gain += diff;
			  }
			    
			  for (i = 0; i < nitems - 1; i++) {
			    pos = si[i].Item.offset + si[i].Item.size;
			    posold = si[i + 1].Item.offset;
			    if ((diff = posold - pos) > 0) {
			      for(
				  fseek(Sfd, posold, SEEK_SET);
				  (nread = fread(buffer, 1, BUFSIZ, Sfd)) > 0;
				  fseek(Sfd, posold, SEEK_SET)
				  ) {
				fseek(Sfd, pos, SEEK_SET);
				fwrite(buffer, 1, nread, Sfd);
				posold += (long)nread;
				pos += (long)nread;
			      }
			      fseek(Sfd, 0, SEEK_END);
			      SSize = ftell(Sfd);
			      ftruncate(fileno(Sfd), (off_t)(SSize - diff));
			      for (j = i + 1; j < nitems; j++) si[j].Item.offset -= diff;
			      gain += diff;
			    }
			  }
			  fseek(Sfd, 0, SEEK_END);
			  pos = (nitems) ? si[nitems - 1].Item.offset + si[nitems - 1].Item.size : UDM_SIG_LEN;
			  posold = ftell(Sfd);
			  if (posold > pos) {
			    ftruncate(fileno(Sfd), (off_t)(pos));
			    gain += (posold - pos);
			  }

			  if (gain > 0) {
			    for (i = 0; i < nitems; i++) {
			      fseek(Ifd, si[i].offset, SEEK_SET);
			      fwrite(&(si[i].Item), sizeof(UDM_STOREITEM), 1, Ifd);
			    }
			    UdmLog(Agent,verb,"Optimize: base 0x%X cleaned, %ld bytes fried", current_base, gain);
			  }
			    
			  UDM_FREE(si);
			}

			CloseBase();
			current_base++;
			current_base %= UdmVarListFindInt(&Conf->Vars, "StoredFiles", 0x10000);
			have_sigalrm = 0;
			alarm((unsigned int) OptimizeInterval);
		}
	}
}


#endif

int main(int argc, char **argv) {

#ifdef HAVE_ZLIB

        fd_set mask;
	struct sockaddr_in server_addr, client_addr;
	const char *config_name= UDM_CONF_DIR "/stored.conf";
	int pid, pid_fd, nport, s, on = 1;
	char pidbuf[1024];
	char buf[1024];
	int log2stderr=1;
	const char *lstn;

	/* Check that another instance isn't running */
	/* and create PID file.                      */

	sprintf(pidname,"%s/%s",UDM_VAR_DIR,"stored.pid");
	pid_fd = open(pidname,O_CREAT|O_EXCL|O_WRONLY,0644);
	if(pid_fd < 0){
		fprintf(stderr,"%s Can't create '%s': %s\n", time_pid_info(), pidname, strerror(errno));
		if(errno == EEXIST){
			fprintf(stderr,"It seems that another stored is already running!\n");
			fprintf(stderr,"Remove '%s' if it is not true.\n",pidname);
 		}
		exit(1);
	}

	sprintf(pidbuf,"%d\n",(int)getpid());
	write(pid_fd,&pidbuf,strlen(pidbuf));

	atexit(&exitproc);
	init_signals();
	UdmInit(); /* Initialize library */

	Conf = UdmEnvInit(NULL);

	UdmLoadConfig(Conf,config_name,0,0);

	StoredFiles = UdmVarListFindInt(&Conf->Vars, "StoredFiles", 0x10000);
	OptimizeInterval = UdmVarListFindInt(&Conf->Vars, "OptimizeInterval", 600);
	OptimizeRatio = UdmVarListFindInt(&Conf->Vars, "OptimizeRatio", 5);

	UdmOpenLog("stored",Conf,log2stderr);

	if(UdmEnvErrCode(Conf)){
 		fprintf(stderr,"%s\n",UdmEnvErrMsg(Conf));
		UdmEnvFree(Conf);
		unlink(pidname);
		exit(1);
	}
	Agent = UdmAgentInit(NULL, Conf, 0);

	UdmLog(Agent, verb, "StoredFiles     : %d (0x%x)", StoredFiles, StoredFiles);
	UdmLog(Agent, verb, "OptimizeInterval: %d sec.", OptimizeInterval);
	UdmLog(Agent, verb, "OptimizeRatio   : %d %%", OptimizeRatio);

	nport = UDM_STORED_PORT;

	if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	  UdmLog(Agent, verb, "%s socket() error %d",time_pid_info(),errno);
	  UdmAgentFree(Agent);
	  UdmEnvFree(Conf);
	  unlink(pidname);
	  exit(1);
	}
	if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on)) != 0){
		UdmLog(Agent, verb, "%s setsockopt() error %d",time_pid_info(),errno);
		UdmAgentFree(Agent);
		UdmEnvFree(Conf);
		unlink(pidname);
		exit(1);
	}

	bzero(&server_addr, sizeof(server_addr));
	server_addr.sin_family = AF_INET;
	if((lstn = UdmVarListFindStr(&Agent->Conf->Vars, "Listen", NULL))) {
	  char * cport;
			
	  if((cport=strchr(lstn,':'))){
	    UdmLog(Agent,verb,"Listening '%s'",lstn);
	    *cport='\0';
	    server_addr.sin_addr.s_addr = inet_addr(lstn);
	    nport=atoi(cport+1);
	  }else{
	    nport=atoi(lstn);
	    server_addr.sin_addr.s_addr = htonl(INADDR_ANY);
	    UdmLog(Agent,verb,"Listening port %d",nport);
	  }
	}else{
	  UdmLog(Agent,verb,"Listening port %d",nport);
	  server_addr.sin_addr.s_addr = htonl(INADDR_ANY);
	}
	server_addr.sin_port = htons((u_short)nport);
	

	if (bind(s, (struct sockaddr *)&server_addr, sizeof(server_addr)) == -1) {
		fprintf(stderr,"%s bind() error %d %s\n",time_pid_info(),errno,strerror(errno));
		unlink(pidname);
		UdmAgentFree(Agent);
		UdmEnvFree(Conf);
		exit(1);
	}

	if (listen(s, UDM_STORED_MAXCLNTS) == -1) {
		fprintf(stderr,"%s listen() error %d %s\n",time_pid_info(),errno,strerror(errno));
		unlink(pidname);
		UdmAgentFree(Agent);
		UdmEnvFree(Conf);
		exit(1);
	}
	
	FD_ZERO(&mask);
	FD_SET(s, &mask);


	if ((pid = fork() ) == -1) {
	  fprintf(stderr,"%s fork() error %d %s\n", time_pid_info(), errno, strerror(errno));
	  unlink(pidname);
	  exit(1);
	}
	if (pid == 0) { /* child process */
	  Optimize();
	  UdmAgentFree(Agent);
	  UdmEnvFree(Conf);
	  exit(0);
	}
	UdmLog(Agent,verb,"Optimization child started.");



	while(1) {
		size_t addrlen;
		int sel;
		struct timeval tval;
		fd_set msk;
	  	
		tval.tv_sec = 300;
		tval.tv_usec = 0;
		msk = mask;
		sel = select(16, &msk, 0, 0, &tval);
		
		if(have_sighup){
			UdmLog(Agent,verb,"SIGHUP arrived. Reloading config.");
			have_sighup=0;
			UdmAgentFree(Agent);
			UdmEnvFree(Conf);
			Conf = UdmEnvInit(NULL);
			
			if(UDM_OK!=UdmLoadConfig(Conf,config_name,0,0)) {
				fprintf(stderr,"%s\n",UdmEnvErrMsg(Conf));
				UdmAgentFree(Agent);
				UdmEnvFree(Conf);
				unlink(pidname);
				exit(1);
			}
			Agent = UdmAgentInit(NULL, Conf, 0);
		}
		if(have_sigint){
			UdmLog(Agent,verb,"SIGINT arrived");
			have_sigint=0;
			break;
		}
		if(have_sigterm){
			UdmLog(Agent,verb,"SIGTERM arrived");
			have_sigterm=0;
			break;
		}
		if(have_sigpipe){
			UdmLog(Agent,verb,"SIGPIPE arrived. Broken pipe !");
			have_sigpipe=0;
			break;
		}
		
		if(sel==0)continue;
		if(sel==-1){
			switch(errno){
				case EINTR:	/* Child */
					break;
			        default:
					UdmLog(Agent,verb,"FIXME select error %d %s",errno,strerror(errno));
			}
			continue;
		}
		
		bzero(&client_addr, addrlen = sizeof(client_addr));
	
	  if ((ns = accept(s, (struct sockaddr *) &client_addr, &addrlen)) == -1) {
	    fprintf(stderr,"%s accept() error %d %s\n",time_pid_info(),errno,strerror(errno));
	    unlink(pidname);
	    exit(1);
	  }

#define Client inet_ntoa(client_addr.sin_addr)

	  UdmLog(Agent, verb, "[%s] Accept", Client);
    
	  if ((pid = fork() ) == -1) {
	    fprintf(stderr,"%s fork() error %d %s\n",time_pid_info(),errno,strerror(errno));
	    unlink(pidname);
	    exit(1);
	  }

	  if (FD_ISSET(s, &msk)) {

	    if (pid == 0) { /* child process */
	      unsigned int rec_id;
	      ssize_t nbytes;
	      long NewItemPos;
	      z_stream zstream;
	    

	      alarm(300); /* 5 min. - maximum time of child execution */
	      init_signals();

	      close(s);
      
	      if ((nbytes = recv(ns, buf, 1, 0))) {
		if (strncmp(buf, "S", 1) == 0) { /* Store document */
		
		  if (UdmRecvall(ns, &rec_id, sizeof(rec_id)) < 0) {
		    closesocket(ns);
		    exit(1);
		  }

		  if (UdmRecvall(ns, &DocSize, sizeof(DocSize)) < 0) {
		    closesocket(ns);
		    exit(1);
		  }
		  Doc = (Byte *) UdmXmalloc(DocSize);
		  if (Doc == NULL) {
		    closesocket(ns);
		    exit(1);
		  }
		  if (UdmRecvall(ns, Doc, DocSize) < 0) {
		    closesocket(ns);
		    exit(1);
		  }

		  UdmSend (ns, "STORED", 6, 0); 
		  close(ns); 

		  /* communications end */

		  zstream.zalloc = Z_NULL;
		  zstream.zfree = Z_NULL;
		  zstream.opaque = Z_NULL;
		  zstream.next_in = Doc;
		
		  if (deflateInit2(&zstream, 9, Z_DEFLATED, 15, 9, Z_DEFAULT_STRATEGY) == Z_OK) {
		
		    zstream.avail_in = DocSize;
		    zstream.avail_out = 2 * DocSize;
		    CDoc = zstream.next_out = (Byte *) UdmXmalloc(2 * DocSize);
		    if (zstream.next_out == NULL) {
		      exit(1);
		    }
		    deflate(&zstream, Z_FINISH);
		    deflateEnd(&zstream);


/* store operations */

		    OpenBase(rec_id, UDM_WRITE_LOCK);



		    if (Item.rec_id == rec_id) {
		      if (Item.size < zstream.total_out) {
			if (fseek(Sfd, 0, SEEK_END)) {
			  ABORT;
			}
			Item.offset = ftell(Sfd);
		      } else {
			if (fseek(Sfd, Item.offset, SEEK_SET)) {
			  ABORT;
			}
		      }
		    } else { /* new rec_id added */
		      if (mishash && Item.rec_id != 0) {
			if (fseek(Ifd, 0, SEEK_END)) {
			  ABORT;
			}
			Item.next = NewItemPos = ftell(Ifd);
			if (fseek(Ifd, CurrentItemPos, SEEK_SET)) {
			  ABORT;
			}
			if (fwrite(&Item, sizeof(UDM_STOREITEM), 1, Ifd) != 1) {
			  ABORT;
			}
			CurrentItemPos = NewItemPos;
		      }
		      Item.rec_id = rec_id;
		      Item.next = 0;
		      if (fseek(Sfd, 0, SEEK_END)) {
			ABORT;
		      }
		      Item.offset = ftell(Sfd);
		    }
		    Item.size = zstream.total_out;
		    if (fwrite(CDoc, zstream.total_out, 1,  Sfd) != 1) {
		      ABORT;
		    }
		    if (fseek(Ifd, CurrentItemPos, SEEK_SET)) {
		      ABORT;
		    }
		    if (fwrite(&Item, sizeof(UDM_STOREITEM), 1, Ifd) != 1) {
		      fprintf(stderr, "Can't write index for file %s\n", Ifilename);
		    }
		    UdmLog(Agent,verb, "[%s] Stored rec_id: %x Size: %d Ratio: %5.2f%%", Client,
			   rec_id, DocSize, 100.0 * zstream.total_out / DocSize);
		    ABORT;
/* /store operations */
		  }

		  ABORT;

		} else if(strncmp(buf, "G", 1) == 0) { /* retrieve document */
		  if (UdmRecvall(ns, &rec_id, sizeof(rec_id)) < 0) {
		    closesocket(ns);
		    exit(1);
		  }
		  
		  OpenBase(rec_id, UDM_READ_LOCK);

		  
		  if (Item.rec_id == rec_id) {
		    if (fseek(Sfd, Item.offset, SEEK_SET)) {
		      UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
		      closesocket(ns); 
		      ABORT;
		    }
		    zstream.avail_in = DocSize = Item.size;
		    zstream.avail_out = UDM_MAXDOCSIZE;
		    CDoc = zstream.next_in = (Byte *) UdmXmalloc(DocSize);
		    Doc = zstream.next_out = (Byte *) UdmXmalloc(UDM_MAXDOCSIZE);
		    if (CDoc == NULL || Doc == NULL) {
		      DocSize = 0;
		      UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
		      closesocket(ns); 
		      ABORT;
		    }
		    zstream.zalloc = Z_NULL;
		    zstream.zfree = Z_NULL;
		    zstream.opaque = Z_NULL;
		    if ((fread(CDoc, DocSize, 1, Sfd) != 1)
			|| (inflateInit2(&zstream, 15) != Z_OK)) {
		      DocSize = 0;
		      UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
		      closesocket(ns); 
		      ABORT;
		    }
		    inflate(&zstream, Z_FINISH);
		    inflateEnd(&zstream);
		    DocSize = zstream.total_out;
		    UdmSend(ns, &DocSize, sizeof(DocSize), 0);
		    UdmSend(ns, Doc, DocSize, 0);
		    closesocket(ns); 
		    
		    
		  } else {
		    UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
		    closesocket(ns); 
		    fprintf(stderr, "[%s] Not found rec_id: %x\n", Client, rec_id);
		    ABORT;
		  }

		  UdmLog(Agent, verb, "[%s] Retrived rec_id: %x Size: %d Ratio: %5.2f%%", Client,
			  rec_id, DocSize, 100.0 * zstream.total_in / DocSize);
		  ABORT;
/***********************/
		} else if(strncmp(buf, "E", 1) == 0) {  /* retrieve by chuncks for excerption */
		  int chunk, i; size_t OldOut;
		  DocSize = 0;

		  if (UdmRecvall(ns, &rec_id, sizeof(rec_id)) < 0) {
		    closesocket(ns);
		    exit(1);
		  }
		  
		  UdmLog(Agent, verb, "[%s] Retrieve by chunks: rec_id: %u", Client, rec_id);

		  OpenBase(rec_id, UDM_READ_LOCK);
		  if (Item.rec_id == rec_id) {
		    if (fseek(Sfd, Item.offset, SEEK_SET)) {
		      DocSize = 0;
		      UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
		      closesocket(ns); 
		      ABORT;
		    }
		    zstream.avail_in = DocSize = Item.size;
		    zstream.avail_out = 0;
		    zstream.zalloc = Z_NULL;
		    zstream.zfree = Z_NULL;
		    zstream.opaque = Z_NULL;
		    CDoc = zstream.next_in = (Byte *) UdmXmalloc(DocSize);
		    Doc = zstream.next_out = (Byte *) UdmXmalloc(UDM_MAXDOCSIZE);
		    if (CDoc == NULL || Doc == NULL) {
		      DocSize = 0;
		      UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
		      closesocket(ns); 
		      ABORT;
		    }

		    if ((fread(CDoc, DocSize, 1, Sfd) != 1)
			|| (inflateInit2(&zstream, 15) != Z_OK)) {
		      DocSize = 0;
		      UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
		      closesocket(ns); 
		      ABORT;
		    }

		    OldOut = 0;
		    for(i = 1; 1; i++) {
		      if (UdmRecvall(ns, &chunk, sizeof(chunk)) < 0) {
			DocSize = 0;
			UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
			closesocket(ns);
			ABORT;
		      }
		      if (chunk == 0) break;
		      zstream.avail_out = UDM_DOCHUNKSIZE;
		      inflate(&zstream, Z_SYNC_FLUSH);
			     
		      DocSize = zstream.total_out - OldOut;
		      UdmSend(ns, &DocSize, sizeof(DocSize), 0);
		      UdmSend(ns, &Doc[OldOut], DocSize, 0);
		      UdmLog(Agent, verb, "[%s] rec_id: %x Chunk %i [%d bytes] sent", Client, rec_id, chunk, DocSize);
		      OldOut = zstream.total_out;
		    }
		    inflateEnd(&zstream);

		  } else {
		    DocSize = 0;
		    UdmSend(ns, &DocSize, sizeof(DocSize), 0); 
		    closesocket(ns); 
		    fprintf(stderr, "[%s] Not found rec_id: %x\n", Client, rec_id);
		    ABORT;
		  }
		  UdmLog(Agent, verb, "[%s] Retrived by chunks rec_id: %x Size: %d Ratio: %5.2f%%", Client,
			 rec_id, zstream.total_out, 100.0 * zstream.total_in / zstream.total_out);
		  ABORT;

/**********************/
		} else if(strncmp(buf, "F", 1) == 0) {  /* check documents presence */
		  int found = 0;
		  if (UdmRecvall(ns, &rec_id, sizeof(rec_id)) < 0) {
		    closesocket(ns);
		    exit(1);
		  }
		  while (rec_id != 0) {
		    OpenBase(rec_id, UDM_READ_LOCK);
		    if (Item.rec_id == rec_id) {
		      found = 1;
		      fprintf(stderr, "[%s] Found rec_id: %x\n", Client, rec_id);
		    } else {
		      found = 0;
		      fprintf(stderr, "[%s] Not found rec_id: %x\n", Client, rec_id);
		    }
		    CloseBase();
		    UdmSend(ns, &found, sizeof(found), 0);

		    if (UdmRecvall(ns, &rec_id, sizeof(rec_id)) < 0) {
		      closesocket(ns);
		      exit(1);
		    }
		  }

		} else if (strncmp(buf, "D", 1) == 0) { /* delete document */
		  long NextItemPos;
		  if (UdmRecvall(ns, &rec_id, sizeof(rec_id)) < 0) {
		    closesocket(ns);
		    exit(1);
		  }
		  OpenBase(rec_id, UDM_WRITE_LOCK);
		  if (Item.rec_id == rec_id) {
		    NextItemPos = Item.next;
		    while(NextItemPos != 0) {
		      if (fseek(Ifd, NextItemPos, SEEK_SET)) {
			ABORT;
		      }
		      if (fread(&Item, sizeof(UDM_STOREITEM), 1, Ifd) != 1) {
			fprintf(stderr, "Can't read hash chain for file %s\n", Ifilename);
			ABORT;
		      }
		      if (fseek(Ifd, CurrentItemPos, SEEK_SET)) {
			ABORT;
		      }
		      CurrentItemPos = NextItemPos;
		      NextItemPos = Item.next;
		      Item.next = CurrentItemPos;
		      if (fwrite(&Item, sizeof(UDM_STOREITEM), 1, Ifd) != 1) {
			fprintf(stderr, "Can't write hash chain for file %s\n", Ifilename);
			ABORT;
		      }
		    }
		    Item.rec_id = 0;
		    Item.offset = 0;
		    Item.next = 0;
		    if (fseek(Ifd, CurrentItemPos, SEEK_SET)) {
		      ABORT;
		    }
		    if (fwrite(&Item, sizeof(UDM_STOREITEM), 1, Ifd) != 1) {
		      fprintf(stderr, "Can't write hash chain for file %s\n", Ifilename);
		      ABORT;
		    }
		    fprintf(stderr, "[%s] Deleted rec_id: %x\n", Client, rec_id);
		  } else {
		    fprintf(stderr, "[%s] rec_id: %x not found for delete\n", Client, rec_id);
		  }
		  CloseBase();

		}else {
		  fprintf(stderr,"%s <hello> error %d %s\n",time_pid_info(),errno,buf /*strerror(errno)*/);
		  close(ns);
		  exit(1);
		}
	      } else {
		fprintf(stderr,"%s recv() <hello> error %d %s\n",time_pid_info(),errno,strerror(errno));
		close(ns);
		exit(1);
	      }


	      close(ns);
	      exit(0);
	    }
	  }
	  /* parent process */
	  close(ns);
	}

	unlink(pidname);
	UdmAgentFree(Agent);
	UdmEnvFree(Conf);

#else
	fprintf(stderr, "zlib support required. Please rebuild with --with-zlib option\n");

#endif
	return 0;
}
